/*
 * The contents of this file are subject to the terms 
 * of the Common Development and Distribution License 
 * (the License).  You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the license at 
 * https://glassfish.dev.java.net/public/CDDLv1.0.html or
 * glassfish/bootstrap/legal/CDDLv1.0.txt.
 * See the License for the specific language governing 
 * permissions and limitations under the License.
 * 
 * When distributing Covered Code, include this CDDL 
 * Header Notice in each file and include the License file 
 * at glassfish/bootstrap/legal/CDDLv1.0.txt.  
 * If applicable, add the following below the CDDL Header, 
 * with the fields enclosed by brackets [] replaced by
 * you own identifying information: 
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * Copyright 2006 Sun Microsystems, Inc. All rights reserved.
 */

package com.sun.enterprise.config.serverbeans.validation.tests;

import java.util.*;
import com.sun.enterprise.util.LocalStringManagerImpl;
import com.sun.enterprise.config.serverbeans.validation.StringManagerHelper;
import com.sun.enterprise.config.serverbeans.validation.Result;

import com.sun.enterprise.admin.util.QuotedStringTokenizer;

public class JvmOptionsTest {

    private static LocalStringManagerImpl smh = 
        StringManagerHelper.getLocalStringsManager();

    private static final JvmOptionsTest instance = new JvmOptionsTest();

    private JvmOptionsTest() {
    }

    public static void validateJvmOptions(String[] jvmOptions, Result result) {
        try {
            instance.checkForNullOptions(jvmOptions);
            Set optionsSet = instance.tokenizeJvmOptions(jvmOptions);
            instance.checkBeginWithDash(optionsSet);
            instance.checkQuotes(optionsSet);
        }
        catch (Exception e) {
            result.failed(e.getMessage());
        }
    }

    private void checkForNullOptions(String[] options) 
        throws InvalidJvmOptionsException
    {
        if (null == options) {
            throw new InvalidJvmOptionsException(getNullJvmOptionMsg());
        }
        for (int i = 0; i < options.length; i++) {
            if (null == options[i]) {
                throw new InvalidJvmOptionsException(getNullJvmOptionMsg());
            }
        }
    }

    private void checkBeginWithDash(Set options) 
        throws InvalidJvmOptionsException
    {
        List invalidOptions = new ArrayList();
        Iterator it = options.iterator();
        while (it.hasNext()) {
            String option = it.next().toString();
            if (!option.startsWith("-") &&
                !(option.startsWith("\"-") && option.endsWith("\""))  ) 
            {
                invalidOptions.add(option);
            }
        }
        if (invalidOptions.size() > 0) {
            throw new InvalidJvmOptionsException(
                getInvalidJvmOptionMsg(invalidOptions.toString()));
        }
    }

    private void checkQuotes(Set options) throws InvalidJvmOptionsException
    {
        List invalidOptions = new ArrayList();
        final Iterator it = options.iterator();
        while (it.hasNext()) {
            String option = it.next().toString();
            if (!checkQuotes(option)) {
                invalidOptions.add(option);
            }
        }
        if (invalidOptions.size() > 0) {
            throw new InvalidJvmOptionsException(
                getIncorrectQuotesMsg(invalidOptions.toString()));
        }
    }

    private boolean checkQuotes(String option)
    {
        int length      = option.length();
        int numQuotes   = 0;
        int index       = 0;

        while (index < length && 
                (index = option.indexOf('\"', index)) != -1)
        {
            numQuotes++;
            index++;
        }
        return ((numQuotes % 2) == 0);
    }

    private Set tokenizeJvmOptions(String[] options) {
        final Set optionsSet = new LinkedHashSet();
        final String DELIM = " \t";
        for (int i = 0; i < options.length; i++) {
            QuotedStringTokenizer strTok = new QuotedStringTokenizer(
                                                options[i], DELIM);
            while (strTok.hasMoreTokens()) {
                optionsSet.add(strTok.nextToken());
            }
        }
        return Collections.unmodifiableSet(optionsSet);
    }

    private String getNullJvmOptionMsg() {
        return smh.getLocalString(getClass().getName() + ".nullJvmOption", 
            "Null Jvm option", new Object[0]);
    }

    private String getInvalidJvmOptionMsg(String invalidOptions) {
        return smh.getLocalString(getClass().getName() + ".invalidJvmOption", 
            "{0} - Invalid Jvm option. Option must begin with -", 
            new Object[] {invalidOptions});
    }

    private String getIncorrectQuotesMsg(String invalidOptions) {
        return smh.getLocalString(getClass().getName() + ".incorrectQuotesInJvmOption", 
            "{0} - Invalid Jvm option. Check quotes", 
            new Object[] {invalidOptions});
    }

    private static final class InvalidJvmOptionsException extends Exception {
        InvalidJvmOptionsException(String msg) {
            super(msg);
        }
    }
}

